package org.msh.tb.bd.tbforms.dhis2.config;

import org.msh.tb.application.tasks.AsyncTaskImpl;
import org.msh.tb.bd.tbforms.dhis2.DHIS2AuthenticationService;
import org.msh.tb.bd.tbforms.dhis2.data.AuthResponse;
import org.msh.tb.bd.tbforms.dhis2.tb10.blockbuilders.TBForm10Block1Builder;
import org.msh.tb.bd.tbforms.dhis2.tb10.blockbuilders.TBForm10Block2Builder;
import org.msh.tb.bd.tbforms.dhis2.tb10.blockbuilders.TBForm10Block5ABuilder;
import org.msh.tb.bd.tbforms.dhis2.tb11.blockbuilders.TBForm11Block3HIVActivitiesBuilder;
import org.msh.tb.bd.tbforms.query.tb11.TB11Block1Row;
import org.msh.tb.bd.tbforms.query.tb11.TB11Block3Row;
import org.msh.tb.bd.tbforms.query.tb12.TB12Row;
import org.msh.tb.entities.enums.TbFormField;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Mauricio on 13/03/2017.
 * Async task that configures all data elements, preparing them to the integration with eTBManager
 */
public class DHIS2DataElementsConfigAsyncTask extends AsyncTaskImpl {

    public static final String PARAM_DHIS2_AUTH_SERVICE = "dHIS2AuthenticationService";
    public static final String PARAM_DHIS2_CONFIG_SERVICE = "dHIS2DataElementConfigService";

    private List<String> dataElementIds;

    private DHIS2AuthenticationService dHIS2AuthenticationService;
    private DHIS2DataElementConfigService dHIS2DataElementConfigService;

    @Override
    protected void starting() {
        dHIS2AuthenticationService = (DHIS2AuthenticationService)getParameter(PARAM_DHIS2_AUTH_SERVICE);
        dHIS2DataElementConfigService = (DHIS2DataElementConfigService)getParameter(PARAM_DHIS2_CONFIG_SERVICE);
    }

    @Override
    protected void execute() {
        try {
            AuthResponse credentials = dHIS2AuthenticationService.auth();
            this.setProgress(8);

            for (int i = 0; i < getDataElementIds().size(); i++) {
                String dataElementId = getDataElementIds().get(i);

                dHIS2DataElementConfigService.configureDataElement(credentials, dataElementId);

                // update progress
                double p = 92.0/getDataElementIds().size();
                p = (p*(i+1))+8.0;
                this.setProgress(new Double(p).intValue());
            }

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void finishing() {
        this.setProgress(100);
    }

    private List<String> getDataElementIds() {
        if (dataElementIds == null) {
            generateIdList();
        }

        return dataElementIds;
    }

    private void generateIdList() {
        dataElementIds = new ArrayList<String>();

        // TB 10 block 1 data elements
        for (TBForm10Block1Builder.DataElementTB10Block1 elem : TBForm10Block1Builder.DataElementTB10Block1.values()) {
            addToDataElementsList(elem.getDhis2Id());
        }

        // TB 10 block 2 data elements
        for (TBForm10Block2Builder.DataElementTB10Block2 elem : TBForm10Block2Builder.DataElementTB10Block2.values()) {
            addToDataElementsList(elem.getDhis2DataElementId());
        }

        // TB 10 block 5A data elements
        for (TBForm10Block5ABuilder.DataElementTB10Block5A elem : TBForm10Block5ABuilder.DataElementTB10Block5A.values()) {
            addToDataElementsList(elem.getDhis2Id());
        }

        // TB 11 block 1
        for (TB11Block1Row elem : TB11Block1Row.values()) {
            addToDataElementsList(elem.getTable1DataElementId());
            addToDataElementsList(elem.getTable2DataElementId());
            addToDataElementsList(elem.getTable3DataElementId());
        }

        // TB 11 block 3
        for (TB11Block3Row elem : TB11Block3Row.values()) {
            addToDataElementsList(elem.getTestedDataElementId());
            addToDataElementsList(elem.getPositiveDataElementId());
        }

        // TB 11 block 3 table 2
        for (TBForm11Block3HIVActivitiesBuilder.TBForm11Block2DataElement elem : TBForm11Block3HIVActivitiesBuilder.TBForm11Block2DataElement.values()) {
            addToDataElementsList(elem.getDataElementId());
        }

        // Tb 12 table 1 and 2 data elements
        for (TB12Row r : TB12Row.values()) {
            addToDataElementsList(r.getTable1DataElementId());
            addToDataElementsList(r.getTable2DataElementId());
        }

        // TB 10 blocks 3, 4, 5B, 6A, 6B, 6C data elements
        // TB 11 Block 2 (IPT) data elements
        for (TbFormField elem : TbFormField.values()) {
            addToDataElementsList(elem.getDhis2DataElementId());
        }
    }

    private void addToDataElementsList(String dataElementId) {
        if (dataElementId != null && !dataElementId.isEmpty()) {
            dataElementIds.add(dataElementId);
        }
    }

    @Override
    public boolean isLogged() {
        return false;
    }
}
